<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Google\Client;
use Google\Service\Gmail;
use Google\Service\Drive;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use RecursiveIteratorIterator;
use RecursiveDirectoryIterator;

class GmailRestoreLocal extends Command
{
    /**
     * The signature of the command.
     * Allows specifying the source backup email and the target restore email.
     */
    protected $signature = 'gmail:restore-local {sourceEmail : The email of the user whose backup you want to restore.} {targetEmail : The email of the user to restore the data to.}';

    protected $description = 'Restores Gmail and Drive from a local backup to a target user.';

    public function handle()
    {
        $sourceEmail = $this->argument('sourceEmail');
        $targetEmail = $this->argument('targetEmail');

        if (!Storage::exists("backups/$sourceEmail")) {
            $this->error("Backup directory for source user '{$sourceEmail}' not found.");
            return 1;
        }

        $this->info("Starting restore from '{$sourceEmail}' to '{$targetEmail}'.");

        $client = new Client();
        $client->setAuthConfig(storage_path('app/google/the-aff-distination-sa.json'));
        // IMPORTANT: Use the new write-access scopes
        $client->setScopes([
            Gmail::GMAIL_MODIFY,
            Drive::DRIVE,
        ]);

        // Restore emails
        // $this->restoreUserEmails($client, $sourceEmail, $targetEmail);

        // Restore Drive files
        $this->restoreUserDrive($client, $sourceEmail, $targetEmail);

        $this->info("Restore process completed for '{$targetEmail}'.");
        return 0;
    }

    private function restoreUserEmails(Client $client, string $sourceEmail, string $targetEmail)
{
    $this->info("-> Starting Gmail restore for: {$targetEmail}");
    try {
        // Impersonate the user who will RECEIVE the emails
        $client->setSubject($targetEmail);
        $gmail = new Gmail($client);

        // Create a label for the restored emails to keep them organized
        $labelName = "Restored - " . date('Y-m-d H:i:s');
        $label = new \Google\Service\Gmail\Label();
        $label->setName($labelName);
        $label->setMessageListVisibility('show');
        $label->setLabelListVisibility('labelShow');
        $createdLabel = $gmail->users_labels->create('me', $label);
        $labelId = $createdLabel->getId();
        
        $this->info("   - Created label '{$labelName}' in target mailbox.");

        $backupPath = "backups/$sourceEmail/gmail";
        $files = Storage::files($backupPath);

        foreach ($files as $file) {
            $rawMessage = Storage::get($file);
            
            $message = new \Google\Service\Gmail\Message();
            $message->setRaw(strtr(base64_encode($rawMessage), '+/', '-_'));

            //
            // --- CHANGE 1: Add 'INBOX' to the label list ---
            // This ensures the message appears in the main inbox.
            //
            $message->setLabelIds([$labelId, 'INBOX']); 

            //
            // --- CHANGE 2: Use import() instead of insert() ---
            // 'import()' allows us to preserve the original date.
            // 'internalDateSource' => 'dateHeader' tells Gmail to use the date from the .eml file.
            //
            $gmail->users_messages->import('me', $message, ['internalDateSource' => 'dateHeader']);
        }
        $this->info("-> Gmail restore SUCCESS for: {$targetEmail}. Found and restored " . count($files) . " emails.");

    } catch (Exception $e) {
        $this->error("-> Gmail restore FAILED for {$targetEmail}. Reason: " . $e->getMessage());
    }
}


    private function restoreUserDrive(Client $client, string $sourceEmail, string $targetEmail)
    {
        $this->info("-> Starting Google Drive restore for: {$targetEmail}");
        try {
            // Impersonate the user who will RECEIVE the files
            $client->setSubject($targetEmail);
            $drive = new Drive($client);

            // Create a main restore folder in the user's Drive to keep things tidy
            $restoreFolderName = "Restore from {$sourceEmail} - " . date('Y-m-d');
            $rootFolder = new \Google\Service\Drive\DriveFile([
                'name' => $restoreFolderName,
                'mimeType' => 'application/vnd.google-apps.folder',
            ]);
            $createdRootFolder = $drive->files->create($rootFolder, ['fields' => 'id']);
            $rootFolderId = $createdRootFolder->getId();
            $this->info("   - Created root restore folder '{$restoreFolderName}' in target Drive.");

            $localBackupPath = storage_path("app/backups/$sourceEmail/drive");
            
            // This is a recursive function to upload folders and files
            $this->uploadDirectoryRecursive($drive, $localBackupPath, $rootFolderId);

            $this->info("-> Google Drive restore SUCCESS for: {$targetEmail}");

        } catch (Exception $e) {
            $this->error("-> Google Drive restore FAILED for {$targetEmail}. Reason: " . $e->getMessage());
        }
    }

    private function uploadDirectoryRecursive(Drive $drive, string $localPath, string $driveParentFolderId)
    {
        $items = scandir($localPath);
        foreach ($items as $item) {
            if ($item == '.' || $item == '..') continue;
            
            $fullLocalPath = $localPath . DIRECTORY_SEPARATOR . $item;
            
            if (is_dir($fullLocalPath)) {
                // It's a directory, create it in Drive and then recurse
                $folder = new \Google\Service\Drive\DriveFile([
                    'name' => $item,
                    'parents' => [$driveParentFolderId],
                    'mimeType' => 'application/vnd.google-apps.folder',
                ]);
                $createdFolder = $drive->files->create($folder, ['fields' => 'id']);
                $this->uploadDirectoryRecursive($drive, $fullLocalPath, $createdFolder->id);
            } else {
                // It's a file, upload it
                $file = new \Google\Service\Drive\DriveFile([
                    'name' => $item,
                    'parents' => [$driveParentFolderId]
                ]);
                $drive->files->create($file, [
                    'data' => file_get_contents($fullLocalPath),
                    'mimeType' => mime_content_type($fullLocalPath),
                    'uploadType' => 'multipart'
                ]);
            }
        }
    }
}
