<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Google\Client;
use Google\Service\Gmail;
use Google\Service\Directory;
use Illuminate\Support\Facades\Storage;
use Google\Service\Drive;
use Exception;

class GmailBackupLocal extends Command
{
    protected $signature = 'gmail:backup-local';
    protected $description = 'Backup all Google Workspace user emails and drives to local storage';

    public function handle()
    {
        $client = new Client();
        $client->setAuthConfig(storage_path('app/google/the-aff-source-sa.json'));
        $client->setScopes([
            Gmail::GMAIL_MODIFY,
            Drive::DRIVE,
            Directory::ADMIN_DIRECTORY_USER_READONLY
            // Add other directory scopes only if you need them for other tasks
        ]);

        //
        // --- PART 1: GET THE LIST OF ALL USERS (as Super Admin) ---
        //
        $this->info("Fetching user list from Google Workspace...");
        $client->setSubject('it@old.the-aff.org'); // Impersonate Super Admin
        $directory = new Directory($client);
        try {
            $users = $directory->users->listUsers(['customer' => 'my_customer'])->getUsers();
            $this->info("Found " . count($users) . " users. Starting backup process...");
        } catch (Exception $e) {
            $this->error("Failed to fetch user list. Is 'azim@momtaz.af' a Super Admin? Error: " . $e->getMessage());
            return 1; // Exit with an error code
        }

        //
        // --- PART 2: LOOP AND BACK UP EACH USER ---
        //
        $this->backupUserDrive($client, 'faryab.fa@old.the-aff.org');
        // foreach ($users as $user) {
        //     $email = $user->getPrimaryEmail();
        //     $this->line("=============================================");
        //     $this->info("Processing user: $email");
        //     $this->line("=============================================");

        //     // Backup emails for the current user
        //     $this->backupUserEmails($client, $email);
            
        //     // Backup drive files for the current user
        //     $this->backupUserDrive($client, $email);
        // }

        $this->info("All backup tasks completed.");
        return 0;
    }

    /**
     * Backs up Gmail for a single user by impersonating them directly.
     */
    private function backupUserEmails(Client $client, string $email)
    {
        $this->info("-> Starting Gmail backup for: {$email}");
        try {
            // CRITICAL: Impersonate the target user for this specific task
            $client->setSubject($email);

            $gmail = new Gmail($client);
            $path = "backups/$email/gmail";
            Storage::makeDirectory($path);
            
            // Use 'me' to refer to the currently impersonated user
            $messagesResponse = $gmail->users_messages->listUsersMessages('me', ['maxResults' => 10]);

            foreach ($messagesResponse->getMessages() as $msg) {
                $message = $gmail->users_messages->get('me', $msg->getId(), ['format' => 'raw']);
                $raw = base64_decode(strtr($message->getRaw(), '-_', '+/'));
                Storage::put("$path/{$msg->getId()}.eml", $raw);
            }
            
            $this->info("-> Gmail backup SUCCESS for: {$email}");
        } catch (Exception $e) {
            $this->error("-> Gmail backup FAILED for {$email}. Reason: " . $e->getMessage());
        }
    }

    /**
     * Backs up Google Drive for a single user by impersonating them directly.
     */
    private function backupUserDrive(Client $client, string $email)
    {
        $this->info("-> Starting Google Drive backup for: {$email}");
        try {
            // CRITICAL: Impersonate the target user for this specific task
            $client->setSubject($email);

            $drive = new Drive($client);
            $basePath = storage_path("app/backups/$email/drive");
            Storage::makeDirectory("backups/$email/drive");

            $folderMap = ['root' => $basePath];
            $pageToken = null;

            do {
                $response = $drive->files->listFiles([
                    'q' => 'trashed = false',
                    'fields' => 'nextPageToken, files(id, name, mimeType, parents)',
                    'pageSize' => 10,
                    'supportsAllDrives' => true,
                    'includeItemsFromAllDrives' => true,
                    'pageToken' => $pageToken,
                ]);

                foreach ($response->getFiles() as $file) {
                    $parent = $file->getParents()[0] ?? 'root';
                    $dir = $folderMap[$parent] ?? $basePath;

                    if ($file->getMimeType() === 'application/vnd.google-apps.folder') {
                        $folderPath = "$dir/" . $this->sanitize($file->getName());
                        $folderMap[$file->getId()] = $folderPath;
                        if (!is_dir($folderPath)) mkdir($folderPath, 0777, true);
                        continue;
                    }

                    $this->downloadDriveFileSmart($drive, $file, $dir);
                }
                $pageToken = $response->getNextPageToken();
            } while ($pageToken);

            $this->info("-> Google Drive backup SUCCESS for: {$email}");
        } catch (Exception $e) {
            $this->error("-> Google Drive backup FAILED for {$email}. Reason: " . $e->getMessage());
        }
    }

    private function downloadDriveFileSmart(Drive $drive, $file, string $dir)
    {
        // This function does not need changes. It correctly uses the provided $drive service.
        if (!is_dir($dir)) mkdir($dir, 0777, true);
        $name = $this->sanitize($file->getName());
        $mime = $file->getMimeType();

        $exportMap = [
            'application/vnd.google-apps.document' => ['mime' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'ext' => 'docx'],
            'application/vnd.google-apps.spreadsheet' => ['mime' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'ext' => 'xlsx'],
            'application/vnd.google-apps.presentation' => ['mime' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation', 'ext' => 'pptx'],
            'application/vnd.google-apps.drawing' => ['mime' => 'image/png', 'ext' => 'png'],
            'application/vnd.google-apps.form' => ['mime' => 'application/zip', 'ext' => 'zip'],
            'application/vnd.google-apps.script' => ['mime' => 'application/vnd.google-apps.script+json', 'ext' => 'json'],
        ];

        try {
            if (isset($exportMap[$mime])) {
                $export = $exportMap[$mime];
                $filename = "$name.{$export['ext']}";
                $response = $drive->files->export($file->getId(), $export['mime'], ['alt' => 'media']);
                file_put_contents("$dir/$filename", $response->getBody()->getContents());
            } elseif (!str_starts_with($mime, 'application/vnd.google-apps')) {
                $response = $drive->files->get($file->getId(), ['alt' => 'media']);
                file_put_contents("$dir/$name", $response->getBody()->getContents());
            } else {
                // Silently skip files that can't be exported, like Google Sites or Jamboard files.
            }
        } catch(Exception $e) {
            $this->warn("   - Could not download file: {$name}. Reason: " . $e->getMessage());
        }
    }

    private function sanitize(string $name): string
    {
        return preg_replace('/[\\/\\\\]/', '_', $name);
    }
}
