<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Google\Client;
use Google\Service\Drive;
use Exception;

class DriveTenantMigration extends Command
{
    /**
     * The signature of the command.
     */
    protected $signature = 'drive:migrate-tenant {sourceEmail} {targetEmail}';

    protected $description = 'Directly migrates Google Drive files from a source user in Tenant A to a target user in Tenant B.';

    // File paths for your two service account keys
    private $sourceKeyPath = 'app/google/the-aff-source-sa.json';
    private $targetKeyPath = 'app/google/the-aff-distination-sa.json';

    public function handle()
    {
        $sourceEmail = $this->argument('sourceEmail');
        $targetEmail = $this->argument('targetEmail');

        $this->info("Starting direct migration from {$sourceEmail} (Tenant A) to {$targetEmail} (Tenant B).");

        // --- Client for Source (Panel A) ---
        $sourceClient = new Client();
        $sourceClient->setAuthConfig(storage_path($this->sourceKeyPath));
        $sourceClient->setScopes([Drive::DRIVE]);
        $sourceClient->setSubject($sourceEmail); // Impersonate source user from the start

        // --- Client for Target (Panel B) ---
        $targetClient = new Client();
        $targetClient->setAuthConfig(storage_path($this->targetKeyPath));
        $targetClient->setScopes([Drive::DRIVE]); // Full drive scope for writing
        $targetClient->setSubject($targetEmail); // Impersonate target user from the start

        // --- Initialize Services ---
        $sourceDrive = new Drive($sourceClient);
        $targetDrive = new Drive($targetClient);

        try {
            // Create a root folder in the destination to keep things organized
            $rootFolderName = "Migrated from {$sourceEmail} - " . date('Y-m-d');
            $rootFolder = new \Google\Service\Drive\DriveFile([
                'name' => $rootFolderName,
                'mimeType' => 'application/vnd.google-apps.folder',
            ]);
            $createdRootFolder = $targetDrive->files->create($rootFolder, ['fields' => 'id']);
            $targetRootFolderId = $createdRootFolder->getId();
            $this->info("Created root folder '{$rootFolderName}' in target Drive.");

            // Start the recursive migration from the root of the source drive
            $this->migrateDirectoryRecursive($sourceDrive, $targetDrive, 'root', $targetRootFolderId);

            $this->info("Migration completed successfully!");

        } catch (Exception $e) {
            $this->error("A critical error occurred: " . $e->getMessage());
            return 1;
        }

        return 0;
    }

    private function migrateDirectoryRecursive(Drive $sourceDrive, Drive $targetDrive, string $sourceParentId, string $targetParentId)
    {
        $pageToken = null;
        do {
            $response = $sourceDrive->files->listFiles([
                'q' => "'{$sourceParentId}' in parents and trashed = false",
                'fields' => 'nextPageToken, files(id, name, mimeType, modifiedTime)',
                'pageSize' => 200, // A smaller page size can be more reliable for memory
                'supportsAllDrives' => true,
                'includeItemsFromAllDrives' => true,
                'pageToken' => $pageToken,
            ]);

            foreach ($response->getFiles() as $sourceFile) {
                $this->line("Processing: {$sourceFile->getName()} ({$sourceFile->getMimeType()})");

                if ($sourceFile->getMimeType() === 'application/vnd.google-apps.folder') {
                    // It's a directory: create it in the destination and recurse
                    $newFolder = new \Google\Service\Drive\DriveFile([
                        'name' => $sourceFile->getName(),
                        'parents' => [$targetParentId],
                        'mimeType' => 'application/vnd.google-apps.folder',
                        'modifiedTime' => $sourceFile->getModifiedTime(), // Set modified time on folder
                    ]);
                    $createdFolder = $targetDrive->files->create($newFolder, ['fields' => 'id']);
                    $this->migrateDirectoryRecursive($sourceDrive, $targetDrive, $sourceFile->getId(), $createdFolder->getId());
                } else {
                    // It's a file: download its content from source and upload to target
                    try {
                        // Get file content from source
                        $content = $sourceDrive->files->get($sourceFile->getId(), ['alt' => 'media']);
                        
                        // Prepare the new file metadata for the target
                        $newFile = new \Google\Service\Drive\DriveFile([
                            'name' => $sourceFile->getName(),
                            'parents' => [$targetParentId],
                            // CRITICAL: Set the original modified time
                            'modifiedTime' => $sourceFile->getModifiedTime(),
                        ]);

                        // Upload the file content to the target
                        $targetDrive->files->create($newFile, [
                            'data' => $content->getBody()->getContents(),
                            'mimeType' => $sourceFile->getMimeType(),
                            'uploadType' => 'multipart'
                        ]);
                    } catch (Exception $e) {
                        $this->warn("   - Could not migrate file: {$sourceFile->getName()}. Reason: " . $e->getMessage());
                    }
                }
            }
            $pageToken = $response->getNextPageToken();
        } while ($pageToken);
    }
}
